	function [h,err] = FRM_REMEZ(order,edge,fx,istage,L,g1,g2,Poles,Delta_p,Delta_s, varargin)
	
	warning off
	%	To be used as a subroutine, in FRM_IIR_FIR, for optimizing FRM filters 
	%	consisting of an odd Cauer filter, i.e., can be realized as a wave
	%	digital lattce filter and two FIR masking filters.
	
	% istage = 1: G1(z) is synthesized:
	% g2(1) == 0: design conventionally
	% g2(1) ~= 0: special technique
	% istage = 2: G2(z) is synthesized:
	% g2(1) == 0: design conventionally
	% g2(1) ~= 0: special technique
	% More comments later!
	%
	%--------------------------------------------------------------------------
	% FRM_REMEZ- A translation of the FORTRAN code of the McClellan-Parks-Rabiner
	% minimax arbitrary-magnitude FIR filter design algorithm [1], referred 
	% in later use to as the MPR code, into MATLAB.
	% Compared with the MPR code, the the peculiarities of the present code are: 
	% (1) The Remez loop has been improved according to [2].
	% (2) Due to the use of MATLAB, the grid pont allocation happened to be 
	%     improved.
	% (3) The desired function is given in a slightly different manner.
	%
	% Inputs to be always used:
	% order - filter order
	% edge - specifies as a fraction of pi the upper and lower edges of the
	%        bands under consideration
	% fx -  specifies the desired values at the edges of each band
	%
	% Inputs to a possible inclusion in varargin: 
	% wtx  - a constant weighting in each band
	% type - three filter types are under consideration, namely,
	%     (1) Filters with symmetric impulse response for multiband design: 
	%       lowpass, higpass, bandpass, and bandstop designs are typical
	%       examples. The benefit of these filters is that the phase is
	%       perfectly linear for all frequencies.
	%     (2) Full band and partial band differentiators: filters with 
	%       antisymmetric impulse response are in use.
	%     (3) Hilbert transformers: again antisymmetry is required.
	%     The distinction between these three types is performed by a 
	%     character array, called type, such that
	%     type(1) = 'm' or type(1) = 'M' = = > 'multiband' design
	%     type(1) = 'd' or type(1) = 'D' = = > 'differentiator'
	%     type(1) = 'h' or type(1) = 'H' = = > 'Hilbert' tranformer
	%     If type is not specified, then type = = 'multiband' is used
	% lgrid - if not specified, lgrid = 16 as in the PM code. To increase the 
	%     accuray of the resulting filter, lgrid = 32, lgrid = 64, 
	%     lgrid = 128,... is recommended. lgrid should be specified as {32},
	%     {64}, {128},... in order to easily find its existance.
	% It is assumed that if wtx, type, and lgrid or two out of them are in use, 
	% then they are specified in the above order.
	%--------------------------------------------------------------------------
	% Compared with the MPR code, there are two differences in the input data:
	% First, in the MPR code, lgrid is specified inside the code and, secondly,
	% in the present code, the desired function, instead of being a constant as
	% in the MPR code, is given at both band edges meaning that the desired 
	% function is a straight line passing two x-y coordinates. This option has 
	% been introduced after following the trials of some MATLAB people to 
	% translate the MPR code into MATLAB. Perhaps, those MATLAB people could not
	% understand that for differentiators the desired function is a straight 
	% line. Anyway, this option is now in use!
	%------------------------------------------------------------------------- 
	% ######################################################################
	% Example calls and comparisons with the firpm function here.
	% ####################################################################
	%
	% Outputs
	%   h - coefficients of the filter
	%   err - the final absolute value of the weighted error function
	%
	% References:
	% [1] J. H. McClellan, T. W. Parks, and L. R. Rabiner, "A computer program 
	%   for designing optimum FIR linear phase digital filters," IEEE 
	%   Transactions on Audio and Electroacoustics, vol. AU-21, no. 6, pp.
	%   506 - 526, December 1973; 
	% - also reprinted in IEEE Acoustics, Speech, and Signal Processing 
	%   Society. Digital Signal Processing Committee, eds., "Selected 
	%   Papers in Digital Signal Processing, II", pp. 97 - 117, IEEE Press, 
	%   New York, 1976; 
	% - the FORTRAN program itself and its short description is included in 
	%   IEEE Acoustics, Speech, and Signal Processing Society. Digital 
	%   Signal Processing Committee, eds., "Programs for Digital Signal 
	%   Processing", pp. 5.1-1 - 5.1-13, IEEE Press, John Wiley & Sons 
	%   New York, 1979.
	% [2] M. Ahsan and T. Saramki, "A MATLAB based optimum multiband FIR 
	%   filters design program following the original idea of the Remez 
	%   multiple exchange algorithm," in Proc. 2011 IEEE International 
	%   Symposium on Circuits and Systems, Rio de Janeiro, Brazil, 
	%   May 15-17, 2011, pp. 137 - 140. 
	%
	% ==============================================================
	% Check varargin according to the above assumptions
	% =============================================================
	%varargin
	lgrid = 16; % default value
	nn = length(varargin);
	if nn > 0 & iscell(varargin{nn})
		lgrid = varargin{nn}{:};
		nn = nn-1;
		varargin = varargin(1:nn);
	end
	if nn == 1
		if ischar(varargin{1})
			type = varargin{1};
			wtx = ones(1,length(fx)/2);
		else
			wtx = varargin{1};
			type = 'multiband';
		end
	elseif nn == 2
		wtx = varargin{1};
		type = varargin{2};
	else
		wtx = ones(1, length(fx)/2);
		type = 'multiband';
	end
	% ==============================================================
	% Find out jtype that was used in the PM code.
	% This not necessary but simplifies the undertanding of this code snippet.
	% ==============================================================
	if type(1) == 'd' | type(1) == 'D'
		jtype = 2;	% Differentiator
	elseif type(1) == 'h' | type(1) == 'H'
		jtype = 3;	% Hilbert transformer 
	elseif type(1) == 'm' | type(1) == 'M'
		jtype = 1;	% Multiband filter
	else
		jtype = 1;	% Multiband filter
	end
	% ==============================================================
	% Determine the filter cases and nfcns, the number of basis functions to be 
	% used in the Remez algorithm 
	% In the below, filtercase = 1,2,3,4 is used for making it easier to 
	% understand this code snippet.  
	% ==============================================================
	if jtype == 1
		if rem(order,2) == 0 
			filtercase = 1; % Even order and even symmetry; multiband filter
		else
			filtercase = 2; % Odd order and even symmetry; multiband filter
		end
	else
		if rem(order,2) == 0
			filtercase = 3; % Even order and odd symmetry; a Hilbert transfor-
			% er or a differentiator (jtype indicates)
		else
			filtercase = 4; % Odd order and odd symmetry; a Hilbert transformer
			% or a differentiator (jtype indicates)
		end
	end
	nfcns = fix((order+1)/2);
	if filtercase == 1, nfcns = nfcns + 1; end
	% nfcns = order/2+1 for filtercase = 1  no fixed zeros
	% nfcns = (order+1)/2 for filtercase = 2; fixed zero at z = -1 ==> 
	%                   fixed term (1+z^{-1})/2
	% nfcns = order/2 for filtercase = 3;   fixed zeros at z = 1 and z = -1 ==> 
	%                   fixed term (1-z^{-2})/2
	% nfcns = (order+1) for filtercase = 4;  fixed zero at z = -1 ==> 
	%                   fixed term (1-z^{-2})/2
	% ==============================================================
	% DETERMINE grid, des, and wt 
	% ==============================================================
	% Compared with the MPR code, there are the following key differences:
	% (1) The upper edge for each band under consideration is automatically 
	%   included in grid. This somehow increases the accuracy. 
	% (2) Since the frequency range is now from 0 to 1, delf has been increased
	%   by a factor of 2.
	% (3) The change of des and wt depending on the filter type is peformed 
	%   before using the (modified) Remez algorithm.
	% (4) The removal of problematic angular frequencies at 0 and pi is 
	%   performed simultaneously for all filter types. Now the remomal is
	%   is performed while generating grid.
	% ==============================================================
	nbands = length(edge)/2; delf = 0.5/(lgrid*nfcns); delf = 2*delf;
	grid = []; des = []; wt = [];
	ll = 1;
	while ll <= nbands
		number_grid = ceil((edge(2*ll)-edge(2*ll-1))/delf);
		grid_more = linspace(edge(2*ll-1),edge(2*ll),number_grid+1);
		% TEST whether there exists harmful omega = 0 or omega = pi or both
		if ll == 1 & (filtercase == 3 | filtercase == 4) & grid_more(1) < delf
			grid_more(1) = []; number_grid = number_grid-1;
		end	
		if ll == nbands & (filtercase == 2 | filtercase == 3) & grid_more(end) > 1-delf
			grid_more(end) = []; number_grid = number_grid-1;
		end
		grid = [grid grid_more]; 
		if jtype ~= 2
			wt_more = wtx(ll)*ones(1,number_grid+1); 
			if fx(2*ll) ~= fx(2*ll-1) 
				des_more = linspace(fx(2*ll-1),fx(2*ll),number_grid+1);
			else
				des_more = fx(2*ll)*ones(1,number_grid+1);
			end
		else
			des_more = fx(2*ll)*grid_more*pi;
			if abs(fx(2*ll-1)) < 1.0e-3
				wt_more = wtx(ll)*ones(1,number_grid+1);
			else
				wt_more = wtx(ll)./(grid_more*pi);
			end   
		end
		des = [des des_more];	wt = [wt wt_more];
		ll = ll+1;
	end
	% ==============================================================
	% Compared the MPR code, there are two basic differences:
	% (1) In the allocation of grid points in each band, delf_new has been 
	%   predetermined such that the last point automatically occurs at the 
	%   upper edge. In the original code, it becomes, due to the use of delf, 
	%   in most cases larger than the edge. As a consequence, the last point
	%   is removed and the value of previous grid point is fixed to take on
	%   the edge value. Hence, the distance within the last and second last
	%   grid points is in most cases larger than delf.
	% (2) In the jtype = 2 case, des = pi*grid*fx, instead of des = grid*fx, and
	%   wt = wtx/(pi*grid), instead of wt = wtx/grid, are used provided that 
	%   fx > = 1.0e-3.
	% ==============================================================
	% Modify des and wt depending on the filter case
	% ==============================================================
	if filtercase == 2
		des = des./cos(pi*grid/2); % The response of the fixed term 
		wt = wt.*cos(pi*grid/2);  % (1+z^{-1})/2 is cos(omega/2)
	end
	if filtercase == 4
		des = des./sin(pi*grid/2); % The response of the fixed term 
		wt = wt.*sin(pi*grid/2);  % (1+z^{-1})/2 is sin(omega/2)
	end
	if filtercase == 3
		des = des./sin(pi*grid);  % The response of the fixed term
		wt = wt.*sin(pi*grid);   % (1+z^{-2})/2 is sin(omega)
	end
	%#########################################################################
	% THE KEY MODIFICATION OF Remez_FIR:
	% Modify des and wt according to istage,L,g1,g2,Poles,Delta
	% ==============================================================
	% separation index between the grid points in the passband and stopband
	% ==============================================================
	ind_cut = find(grid > edge(2) );
	[des,wt] = modify_FRMIIR(pi*grid,istage,L,g1,g2,Poles,Delta_p,Delta_s,ind_cut); 
	if filtercase == 2
		des = des./cos(pi*grid/2); % The response of the fixed term 
		wt = wt.*cos(pi*grid/2);  % (1+z^{-1})/2 is cos(omega/2)
	end
	%##########################################################################
	% ==============================================================
	% CALL THE REMEZ ALGORITHM 
	% ==============================================================
	% Compared with the PM code, there are the following key differences:
	% (1) On purpose, only filters with even order and even symmetry are under
	%   consideration or filtercase = 1 is in use. As a matter of fact, the
	%   original FORTAN code did it indirectly.
	% (2) For achieving this goal, des and wt modified beforehand and the 
	%   possible response for another filter type will be generated later 
	%   based on the present response
	% (3) The Remez multiple exchange loop has been significantly improved.
	% (4) grid, the frequency grid, is now within 0 and 1, instead of being
	%   within 0 and 0.5.
	% ==============================================================
	[h,err,iext] = basicRemez(nfcns,grid,des,wt);
	% ==============================================================
	% Generate the impulse responses for other types 
	% ==============================================================
	nn = length(h);
	if filtercase == 2
		h = [h(1)/2 (h(2:nn)+h(1:nn-1))/2 h(nn)/2]; 
	end
	if filtercase == 3
		h = [h(1)/2 h(2)/2 (h(3:nn)-h(1:nn-2))/2 -h(nn-1)/2 -h(nn)/2]; 
	end
	if filtercase == 4
		h = [h(1)/2 (h(2:nn)-h(1:nn-1))/2 -h(nn)/2]; 
	end
	err = abs(err);
	
	function [h,dev,iext]  = basicRemez(nfcns,grid,des,wt)
	% basicRemez function
	% Inputs
	%   nfcns - number of basis functions 
	%   grid - frequency grid between 0 and 1
	%   des - desired function on frequency grid
	%   wt - weight function on frequency grid
	% Outputs
	%   h - coefficients of the filtercase = 1 filter
	%   dev - the resulting value of the weighted error function
	%   iext - indices of extremal frequencies
	% ==============================================================
	% Some initializations
	% ==============================================================
	ngrid = length(grid); % number of grid points
	l_ove = 1:ngrid; %overall index vector
	temp = (ngrid-1)/nfcns;
	jj = 1:nfcns;
	l_trial = fix([temp*(jj-1)+1 ngrid]);
	nz = nfcns + 1;
	devl = 0;
	niter = 1; 
	itrmax = 250; 
	x_all = cos(pi*grid);
	% ==============================================================
	% Remez loop
	% ==============================================================
	while (niter < itrmax)
		x = cos(pi*grid(l_trial)); 
		%--------------------------------------------------------------------------
		% The following way of calculating the Lagrange interpolation coefficients
		% has been copied from the PM code. The original idea is excellent because
		% the high accuracy is maintained up to the filter order of 2000
		%--------------------------------------------------------------------------
		jet = fix((nfcns - 1)/15) + 1;  
		for mm = 1:nz
			yy = 1;
			for nn = 1:jet
				xx = 2*(x(mm) - x(nn:jet:nz));
				yy = yy*prod(xx(xx ~= 0));
			end
			ad(mm) = 1/yy;
		end
		alter = ones(size(ad));
		alter(2:2:nz) = -alter(2:2:nz);
		dnum = ad*des(l_trial)';
		dden = alter*(ad./wt(l_trial))';
		dev = -dnum/dden; 
		%OMIT
		if abs(dev) <= abs(devl)
			%#############################################
			% a need to use a more informative message.
			%#############################################
			%  warning('convergence problems')
			break;
		end
		devl = dev; 
		y = des(l_trial) + dev*alter./wt(l_trial);
		l_left = setdiff(l_ove,l_trial);
		err_num = zeros(size(l_left)); 
		err_den = err_num; 
		for jj = 1:nz 
			aid = ad(jj)./(x_all(l_left) - x(jj)); 
			err_den = err_den + aid; 
			err_num = err_num + y(jj)*aid;
		end
		wei_err(l_left) = (err_num./err_den - des(l_left)).*wt(l_left);
		wei_err(l_trial) = alter*dev;
		l_aid1 = find(diff(sign(diff([0 wei_err 0]))));
		l_aid2 = l_aid1(abs(wei_err(l_aid1)) >= abs(dev));
		[X,ind] = max(sparse(1:length(l_aid2),... 
		cumsum([1,(wei_err(l_aid2(2:end)) >= 0)...
		~= (wei_err(l_aid2(1:end-1)) >= 0)]), abs(wei_err(l_aid2))));
		l_real_init = l_aid2(ind); 
		if rem(length(l_real_init) - nz,2) == 1 
			if abs(wei_err(l_real_init(1))) <= abs(wei_err(l_real_init(end)))
				l_real_init(1) = []; 
			else
				l_real_init(end) = [];
			end
		end
		while length(l_real_init) > nz
			wei_real = abs(wei_err(l_real_init)); 
			wei_comp = max(wei_real(1:end-1), wei_real(2:end)); 
			if max(abs(wei_err(l_real_init(1))),... 
				abs(wei_err(l_real_init(end)))) <= min(wei_comp)
				l_real_init = l_real_init(2:end-1); 
			else
				[X,ind_omit] = min(wei_comp);
				l_real_init(ind_omit:ind_omit+1) = []; 
			end
		end
		l_real = l_real_init;
		if (l_real == l_trial) 
			break; 
		else
			l_trial = l_real;
			niter = niter + 1; 
		end 
	end 
	iext = l_real; % final indices of the extremal points for the output
	% ==============================================================
	% Generate the impulse response of the filtercase = 1 filter using the IDFT. 
	% It is not very straightforward because the result of the Remez loop is
	% expressed using the Lagrange interpolation formula in barycentric form
	% The generation of this impulse response follows exactly the idea in the 
	% the MPR code.
	% ==============================================================
	cn = 2*nfcns - 1; 
	x_IDFT = 0:2/cn:2*(nfcns-1)/cn;
	x_IDFT = cos(pi*x_IDFT);
	[X,ind1] = intersect(x_IDFT,x); 
	[X,ind2] = intersect(x,x_IDFT);
	ind1 = sort(ind1');
	ind2 = sort(ind2');
	l_ove = 1:nfcns;
	l_left = setdiff(l_ove,ind1);
	num = zeros(size(l_left));
	den = num;
	for jj = 1:nz
		aid = ad(jj)./(x_IDFT(l_left) - x(jj));
		den = den + aid;
		num = num + y(jj)*aid;
	end
	A(l_left) = num./den;
	A(ind1) = y(ind2); 
	for n = 1:nfcns
		h(n) = (1/cn)*(A(1)+sum(2*A(2:nfcns).*cos(2*pi*(1:nfcns-1)*(n-1)/cn)));
	end
	h = real(h);
	h = [h(nfcns:-1:1) h(2:nfcns)];
	 
	% THE THIRD KEY MODIFICATION OF RemezFIR:
	% Use the following two subroutines:
	
	function [des,wt] = modify_FRMIIR(w,istage,L,g1,g2,Poles,Delta_p,Delta_s,ind_cut)
	% More explanations coming up later!
	
	% 	Toolbox for DIGITAL FILTERS USING MATLAB 
	
	%	Author: 			Tapio Saramaki 5.4.2018
	%	Modified by:		LW 2018-06-28
	%	Copyright:			by authors - not released for commercial use
	%	Version:			1	 
	%	Known bugs:			None
	%	Report bugs to:		tapio.saramaki@tut.fi
	
	w_p = w(1:ind_cut-1);
	w_s = w(ind_cut:end);
	if (istage == 1 & g2(1) == 0) | (istage == 2 & g1(1) == 0)
		Adjust_p = zeros(1,length(w_p));
		Adjust_s = zeros(1,length(w_s));
		Den_p = ones(1,length(w_p));
		Den_s = ones(1,length(w_s));
	elseif istage == 1 & g2(1) ~= 0
		Adjust_p = (1-FIRres(g2,w_p).^2).*(1-abs(IIRres(Poles,L*w_p)).^2);
		Adjust_s = FIRres(g2,w_s).^2.*(1-abs(IIRres(Poles,L*w_s)).^2);
		Den_p = abs(IIRres(Poles,L*w_p)).^2;
		Den_s = abs(IIRres(Poles,L*w_s)).^2;
	elseif istage == 2 & g1(1) ~= 0
		Adjust_p = (1-FIRres(g1,w_p).^2).*abs(IIRres(Poles,L*w_p)).^2;
		Adjust_s = FIRres(g1,w_s).^2.*abs(IIRres(Poles,L*w_s)).^2;
		Den_p = 1-abs(IIRres(Poles,L*w_p)).^2; 
		Den_s = 1-abs(IIRres(Poles,L*w_s)).^2;
	end
	small = 10^(-30);
	ind = Den_p < small;
	Den_p(ind) = small;
	ind = Den_s < small;
	Den_s(ind) = small;
	values_p = 1-(Delta_p-Adjust_p)./Den_p;
	ind = values_p < small;
	values_p(ind) = small;
	low_p = sqrt(values_p);
	upp_p = ones(1,length(w_p));
	low_s = -sqrt((Delta_s-Adjust_s)./Den_s);
	upp_s = +sqrt((Delta_s-Adjust_s)./Den_s);
	low = [low_p low_s];
	upp = [upp_p upp_s];
	des = (low+upp)/2.;
	wt = 2./(upp-low);
	if istage == 1,
		break
	end

	function [ZPFR] = FIRres(h,wT)
	% After knowing the impulse response included in h for a linear-phase FIR 
	% filter (symmtric or antisymmetric response),[ZPR] = zeroamw(h,wT) 
	% evaluates the zero-phase response (ZPR) of the filter at angular 
	% frequencies wT
	
	% 	Toolbox for DIGITAL FILTERS USING MATLAB 
	
	%	Author: 			Tapio Saramaki 17.11.1997
	%	Modified by:		LW 2018-06-28
	%	Copyright:			by authors - not released for commercial use
	%	Version:			1	 
	%	Known bugs:			None
	%	Report bugs to:		tapio.saramaki@tut.fi
	
	N = length(h);	NN = floor((N-1)/2);	
	if 2*round(N/2) == N; iodd = 1; else iodd = 0; end
	sum1 = sum(h(1:NN));sum2 = sum(h(N+1-NN:N));
	if abs(sum2-sum1) < 0.5*abs(sum1), isy = 0; else isy = 1; end
	if iodd == 0 & isy == 0; itype = 1;end
	if iodd == 1 & isy == 0; itype = 2;end
	if iodd == 0 & isy == 1; itype = 3;end
	if iodd == 1 & isy == 1; itype = 4;end
	L = floor((N+2)/2);
	ZPFR = zeros(size(wT));
	if itype == 1
		ZPFR = ZPFR+h(L);
	end
	for k = 1:L-1
		if itype == 1
			ZPFR = ZPFR+2*h(L-k)*cos(k*wT);
		end
		if itype == 2
			ZPFR = ZPFR+2*h(L-k)*cos((k-0.5)*wT);
		end
		if itype == 3
			ZPFR = ZPFR+2*h(L-k)*sin(k*wT);
		end
		if itype == 4
			ZPFR = ZPFR+2*h(L-k)*sin((k-0.5)*wT);
		end
	end

	function [response] = IIRres(P, wT)	
	% Evaluates the frequency response of two allpass filters in a lowpass 
	% (LP)- highpass (HP) lattice wave digital (LWD) filter pair at angular 
	% frequencies wT.
	
	% 	Toolbox for DIGITAL FILTERS USING MATLAB 
	
	%	Author: 			Tapio Saramaki 8.5.2018
	%	Modified by:		LW 2018-06-28
	%	Copyright:			by authors - not released for commercial use
	%	Version:			1	 
	%	Known bugs:			None
	%	Report bugs to:		tapio.saramaki@tut.fi
	
	P1 = P(1:2:length(P)); 		P2 = P(2:2:length(P));
	expwT = exp(j*wT);
	S1 = -ones(1,length(wT));	
	for n = 1:length(P1)
		S1 = S1.*(-P1(n)+expwT)./(1-P1(n)*expwT);
	end
	S2 = ones(1,length(wT));
	for n = 1:length(P2)
		S2 = S2.*(-P2(n)+expwT)./(1-P2(n)*expwT);
	end
	response = abs((S2-S1)/2);
	
	
		
